# 🌐 Website Setup Guide

Deploy the attendance system to your web server.

## Files Included

```
attendance100/
├── index.php          - Dashboard (view attendance)
├── webhook.php        - Receives device data
├── README.md          - Quick start guide
├── DEVICE_SETUP.md    - Device configuration guide
└── WEBSITE_SETUP.md   - This file
```

`attendance.json` will be auto-created.

---

## Installation

### Method 1: Upload via FTP/SFTP

1. **Connect to server:**
   ```
   Host: dev.ntranepal.edu.np
   Protocol: SFTP or FTP
   ```

2. **Navigate to:**
   ```
   /public_html/
   or
   /var/www/html/
   ```

3. **Create folder:**
   ```
   attendance100/
   ```

4. **Upload files:**
   - index.php
   - webhook.php
   - README.md
   - DEVICE_SETUP.md
   - WEBSITE_SETUP.md

### Method 2: Upload via cPanel

1. Login to cPanel
2. Go to **File Manager**
3. Navigate to `public_html/`
4. Click **New Folder** → Name: `attendance100`
5. Enter folder
6. Click **Upload**
7. Select and upload all files

### Method 3: Command Line (SSH)

```bash
# Navigate to web root
cd /var/www/html/

# Create directory
mkdir attendance100

# Upload files (use your method)
# Then set permissions:
cd attendance100
chmod 755 .
chmod 644 *.php
```

---

## Set Permissions

### Option A: Via SSH
```bash
cd /path/to/attendance100/
chmod 755 attendance100/
chmod 644 index.php webhook.php
touch attendance.json
chmod 666 attendance.json
```

### Option B: Via FTP Client
- Folder `attendance100/`: `755`
- Files `index.php`, `webhook.php`: `644`
- File `attendance.json`: `666` (will be auto-created if not exists)

### Option C: Via cPanel
1. Right-click file → **Change Permissions**
2. Set as above

---

## Test Installation

### Test 1: Dashboard Access

Open browser:
```
https://dev.ntranepal.edu.np/attendance100/
```

**Expected:** Dashboard page with empty table or stats showing zeros.

**If error:** Check file permissions and PHP version.

### Test 2: Webhook Access

Open browser or use curl:
```bash
curl -X GET https://dev.ntranepal.edu.np/attendance100/webhook.php
```

**Expected:** JSON response:
```json
{
  "status": 405,
  "message": "Method not allowed"
}
```

This is correct! Webhook only accepts POST.

### Test 3: Webhook POST Test

Create test file `test.php` in attendance100/:
```php
<?php
$data = [
    'event_log' => json_encode([
        'dateTime' => date('Y-m-d\TH:i:s'),
        'ipAddress' => '192.168.1.73',
        'macAddress' => 'a4:d5:c2:14:dd:73',
        'AccessControllerEvent' => [
            'majorEventType' => 5,
            'subEventType' => 75,
            'employeeNoString' => 'emp999'
        ]
    ])
];

$ch = curl_init('https://dev.ntranepal.edu.np/attendance100/webhook.php');
curl_setopt($ch, CURLOPT_POST, 1);
curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);

echo "Response: " . $response . "\n";
```

Run: `php test.php` or visit: `https://dev.ntranepal.edu.np/attendance100/test.php`

**Expected:** Success response with attendance ID.

---

## Verify Data Storage

### Check attendance.json exists:

```bash
ls -la attendance.json
```

**Expected:** File exists with write permissions.

### Check content:

```bash
cat attendance.json
```

**Expected:** JSON array with records (if any test data sent).

---

## PHP Requirements

### Minimum Requirements:
- **PHP Version**: 7.4 or higher
- **Extensions**: json, date
- **Functions**: file_get_contents, file_put_contents

### Check PHP Version:
```bash
php -v
```

or create `info.php`:
```php
<?php phpinfo();
```

Visit: `https://dev.ntranepal.edu.np/attendance100/info.php`

*(Delete after checking)*

---

## Web Server Configuration

### Apache (.htaccess)

If you need custom rules, create `.htaccess`:

```apache
# Enable error reporting (remove in production)
php_flag display_errors on
php_value error_reporting E_ALL

# Set timezone
php_value date.timezone "Asia/Kathmandu"

# Protect JSON files
<FilesMatch "\.json$">
    Order allow,deny
    Deny from all
</FilesMatch>

# Allow webhook POST
<Files "webhook.php">
    Order allow,deny
    Allow from all
</Files>
```

### Nginx

Add to server block:

```nginx
location /attendance100/ {
    index index.php;
    try_files $uri $uri/ =404;
    
    location ~ \.php$ {
        include fastcgi_params;
        fastcgi_pass unix:/var/run/php/php-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
    }
    
    location ~ \.json$ {
        deny all;
    }
}
```

---

## Security Checklist

### Production Security:

- [ ] Change file permissions (644 for PHP, 666 for JSON)
- [ ] Block direct access to attendance.json
- [ ] Add IP whitelist to webhook (optional)
- [ ] Enable HTTPS only
- [ ] Add authentication to index.php
- [ ] Regular backups of attendance.json
- [ ] Monitor file size

### Optional: IP Whitelist

Add to webhook.php (top):
```php
$allowedIPs = ['192.168.1.73']; // Device IP
if (!in_array($_SERVER['REMOTE_ADDR'], $allowedIPs)) {
    http_response_code(403);
    die(json_encode(['status' => 403, 'message' => 'Forbidden']));
}
```

---

## Backup Strategy

### Manual Backup:

```bash
cd /path/to/attendance100/
cp attendance.json "backup-$(date +%Y%m%d-%H%M%S).json"
```

### Automated Daily Backup (Cron):

```bash
# Edit crontab
crontab -e

# Add line (runs daily at 2 AM)
0 2 * * * cd /path/to/attendance100 && cp attendance.json "backups/backup-$(date +\%Y\%m\%d).json"
```

Create backups folder:
```bash
mkdir /path/to/attendance100/backups
chmod 755 backups/
```

### Download Backup:

Via FTP or:
```bash
scp user@dev.ntranepal.edu.np:/path/to/attendance100/attendance.json ./local-backup.json
```

---

## Monitoring

### Check Logs:

```bash
# Apache error log
tail -f /var/log/apache2/error.log | grep attendance100

# Nginx error log
tail -f /var/log/nginx/error.log | grep attendance100

# PHP errors (if enabled)
tail -f /path/to/attendance100/error.log
```

### Monitor File Size:

```bash
ls -lh attendance.json
```

System auto-limits to 5000 records (~500KB-1MB).

---

## Troubleshooting

### Issue: 404 Not Found

**Solutions:**
- Verify files uploaded correctly
- Check folder name is exactly `attendance100`
- Check web server is running
- Verify document root path

### Issue: 500 Internal Server Error

**Solutions:**
- Check PHP errors: `tail -f /var/log/apache2/error.log`
- Verify PHP version: `php -v` (need 7.4+)
- Check file permissions
- Look for syntax errors: `php -l index.php`

### Issue: attendance.json not created

**Solutions:**
- Check folder write permissions: `chmod 755 attendance100/`
- Manually create: `touch attendance.json && chmod 666 attendance.json`
- Check disk space: `df -h`
- Check SELinux/AppArmor permissions

### Issue: Empty dashboard

**Solutions:**
- Check attendance.json exists and has data
- Verify webhook receiving data (test with curl)
- Check PHP can read file: `ls -la attendance.json`
- Test manually: add data to attendance.json

---

## Maintenance

### Daily:
- [ ] Check dashboard loads correctly
- [ ] Verify new attendance appearing

### Weekly:
- [ ] Backup attendance.json
- [ ] Check file size
- [ ] Review for errors

### Monthly:
- [ ] Archive old data
- [ ] Check server logs
- [ ] Update documentation if needed

---

## URLs Reference

| Purpose | URL |
|---------|-----|
| Dashboard | https://dev.ntranepal.edu.np/attendance100/ |
| Webhook | https://dev.ntranepal.edu.np/attendance100/webhook.php |
| README | https://dev.ntranepal.edu.np/attendance100/README.md |

---

## Quick Commands

```bash
# Check installation
cd /path/to/attendance100 && ls -la

# View logs
tail -50 attendance.json | jq

# Check permissions
ls -la attendance100/

# Test PHP
php -v
php -m | grep json

# Test webhook
curl -X POST https://dev.ntranepal.edu.np/attendance100/webhook.php

# Backup
cp attendance.json backup-$(date +%Y%m%d).json

# View recent entries
tail -20 attendance.json | jq
```

---

## Support

If issues persist:

1. Check PHP error logs
2. Verify file permissions
3. Test with curl command
4. Check attendance.json manually
5. Review device logs

---

## Next Steps

After setup complete:

1. ✅ Files uploaded
2. ✅ Permissions set
3. ✅ Dashboard accessible
4. ✅ Webhook tested
5. → Configure device (see DEVICE_SETUP.md)
6. → Test with actual attendance
7. → Monitor for 24 hours
8. → Extend with MVC if needed
